/****************************************************************
* Firmware of the TI TRF7970A EVM Rev. 0
* 02.Dec.2010
* Copyright (C) 2010 Texas Instruments, Inc. 
*
* DESCRIPTION: The TRF797x is an integrated analog front end and
* data framing system for a 13.56 MHz RFID reader system.
* Built-in programming options make it suitable for a wide range
* of applications both in proximity and vicinity RFID systems.
* The reader is configured by selecting the desired protocol in
* the control registers. Direct access to all control registers
* allows fine tuning of various reader parameters as needed.
*
* The TRF796x is interfaced to a MSP430F2370 microcontroller
* through a parallel interface or a  SPI (serial) interface using
* hardware Usart. The MCU is the master device and initiates all
* communication with the reader. The interface is selected by
* jumper setting.
*
* The anti-collision procedures (as described in the ISO
* standards 14443A/B and 15693) are implemented in the MCU
* firmware to help the reader detect and communicate with one
* PICC/VICC among several PICCs/VICCs.
*
* Initial the Firmware is running in stand alone mode. Found tags
* are displayed by LEDs. If a host is detected, the Firmware will
* switch to remote mode by remote_flag.
*
* The MCU communicates with the higher level host station
* (normally a PC) through a UART. The user can send the desired
* commands and the MCU interprets the data received and sends
* appropriate commands to the TRF796x.
* 
*
* AUTHOR:	Alexander Kozitsky 	DATE: 2-21-2013
*           alex.kozitsky@ti.com
*
* CHANGES:
* REV.	DATE		WHO	DETAIL
* 00	02Dec2010	RP	Orginal Version
* 01	07Dec2010	RP	Changed SPI clock frequency in spi.c
* 02    02Feb2013   AK  Preliminary version of the proximity tag detector
* BUILT WITH:
* Code Composer Studio Core Edition Version: 4.1.3.00038
* (C) Copyright Texas Instruments, 2009. All rights reserved.
*****************************************************************
*****************************************************************
* All software and related documentation is provided "AS IS" and
* without warranty or support of any kind and Texas Instruments
* expressly disclaims all other warranties, express or implied,
* including, but not limited to, the implied warranties of
* merchantability and fitness for a particular purpose. Under no
* circumstances shall Texas Instruments be liable for any
* incidental, special or consequential damages that result from
* the use or inability to use the software or related
* documentation, even if Texas Instruments has been advised of
* the liability.
*
* Unless otherwise stated, software written and copyrighted by
* Texas Instruments is distributed as "freeware". You may use
* and modify this software without any charge or restriction.
* You may distribute to others, as long as the original author
* is acknowledged.
****************************************************************/

//===============================================================
// Program with hardware USART and paralel communication        ;
// interface with TRF reader chip.                              ;
//                                                              ;
// PORT4.0 - 4.7 - (IO0-IO7) for parallel interface             ;
//                           with reader chip                   ;
// PORT3.0 - PORT3.3 - USCI_B0 ---> SPI interface with reader   ;
// PORT3.4 - PORT3.5 - USCI_A0 ---> USB/UART control signals    ;
// PORT2.1 - IRQ                                                ;
// PORT3.3 - DATA_CLK                                           ;
// PORT1.7 - PORT1.3 - signaling LEDs                           ;
// PORT1.0 - TRF Enable                                         ;
//===============================================================

#include <stdlib.h>			// general purpose standard library
#include <stdio.h>			// standard input/output header
#include "host.h"
#include "iso14443a.h"
#include "iso14443b.h"
#include "iso15693.h"
#include "felica.h"
#include "mcu.h"
#include "trf797x.h"
#include "types.h"
#include "uart.h"
#include "Dynamic_Calibration.h"
#include "main.h"

//===============================================================

u08_t buf[300];

u08_t enable = 0;

u08_t i_reg = 0x01;							// interrupt register

u08_t irq_flag = 0x00;
u08_t rx_error_flag = 0x00;
s16_t rxtx_state = 1;							// used for transmit receive byte count

s16_t nfc_state;

u08_t remote_flag = 0;
u08_t stand_alone_flag = 1;
u08_t reader_mode = 0x00;						// determines how interrupts will be handled


u16_t SniffCardTimeThreshold = 0;
u16_t WakeupTime = 0;
u16_t SleepNumberOfCycles = 0;
u08_t TagDetected = 0;
u08_t Record[20];

void main (void)
{
	u16_t time;

	WDTCTL = WDTPW + WDTHOLD;				// stop watchdog timer
	
	#ifdef ENABLE_HOST
	#ifdef UART_PRINTF  					// check could be removed if no debugging needed
		UartSetup();
	#endif
	#endif

	ENABLE_SET;                         	// P1.0 is switched in output direction
	TRF_DISABLE;                        	// Keep the TRF disabled

	while(1)                                // infinite loop
    {
        McuOscSel(DCO_CLOCK_SOURCE);        // we will power down the TRF7970A into Power Down Mode 0 - (total power down) no clock source will be available to run the MSP430 thus the internal DCO is needed
        TRF_DISABLE;                        // Put the TRF into shutdown mode, very little power consumption.  No memory is retained in this state, Running at 8 Mhz
        SetupSleepState ();                 // prepare the device for low power state
        SetWakeupTimer (SYSTEM_SLEEP_TIME, ACLK_CLOCK); // clock will be ACLK (at 12khz), 3996 cycles or ~.333 seconds
        __bis_SR_register(LPM3_bits + GIE); // Timer_B will wake-up the MCU
        SetupWakeState ();                  // reset the port settings for operation
    	Trf797xReConfig();					// reinitialize the SPI module
        TRF_ENABLE;                         // enable the TRF from low power state into active
		SetWakeupTimer (400, SMCLK_CLOCK);  // sleep 50uS
		__bis_SR_register(LPM0_bits + GIE); // Timer_B will wake-up the MCU
        SniffInitialSettings();             // Reinitialize the TRF7970A
        SetWakeupTimer (35, ACLK_CLOCK);    // 2.9ms sleep to allow the TRF79x0A to initialize (mainly the crystal)
        __bis_SR_register(LPM3_bits + GIE); // Timer_B will wake-up the device
		InitForCardSniff();                 // Initialization for card sniffing
        __delay_cycles(50);					// give time for the clock change to stabilize
        McuOscSel(TRF_CLOCK_SOURCE);        // set up the system for clock source of the TRF79x0A- running at 13.56MHz
		time = ComparatorSlopeTime();       // Pulse the transmitter and record the time the signal crosses the threshold voltage

        if (time == 0)
        {									// this is the case where a timeout occurred in the ComparatorSlopeTime()
        	continue; 						//return to the start
        }

        // this block only executes on startup and not afterwards
        // it is for initial calibration
        if (Initial_Calibration (time))
        {
            continue; 						//initial calibration is not complete
        }

#ifdef USE_AUTOMATIC_CALIBRATION
        if (Automatic_Calibration(time, THRESHOLD_OFFSET))		    // does automatic calibration indicate that a threshold was exceeded and a read should occur?
#else
        if (time >= SniffCardTimeThreshold)							// without automatic calibration, useful for testing ranges
#endif
        {
            LED_OPEN1_ON;                                           // Assert an LED to indicate that the card has been detected (TAG-IT LED)
            ReInitForStandardOp();                                  // Init the TRF transciever for standard operation
            #if TRIGGER						// in Mcu.h
                LED_OPEN1_ON;
                McuDelayMillisecond(1);
                LED_OPEN1_OFF;
            #endif
            if(remote_flag == 1) 			// if in remote mode
            {								// remote mode can be disabled in host.h
                #ifdef ENABLE_HOST
                    buf[4] = 0xff;			// "TRF7970 EVM" message in GUI
                    HostCommands();			// device works according host commands
                #endif
            }
            else
            {
            	// enable the card type needed, defaults to ISO14443A
                #ifdef ENABLE15693				// this standard can be disabled in ISO15693.h
                    Iso15693FindTag();			// detects ISO15693 in stand-alone mode
                #endif
                if(remote_flag == 0)
                {
                    #ifdef ENABLE14443A			// this standard can be disabled in ISO14443A.h
                        Iso14443aFindTag();		// detects ISO14443A in stand-alone mode
                    #endif
                }
                if(remote_flag == 0)
                {
                    #ifdef ENABLE14443B			// this standard can be disabled in ISO14443B.h
                        Iso14443bFindTag();		// detects ISO14443B in stand-alone mode
                    #endif
                }
                if(remote_flag == 0)
                {
                    #ifdef ENABLE_FELICA		// this standard can be disabled in felica.h
                        FindFelica();			// detects FeliCa in stand-alone mode
                    #endif
                }
                __no_operation();
            }
        }
        else
        {
            LED_OPEN1_OFF;                      // No card has been detected in the field
        }
	}
}

// Used for initial calibration, no detections or reads possible during this phase
u08_t Initial_Calibration (u16_t time)
{
	static u08_t cal_set = 0;
	u08_t ret_val = 0;

    if (cal_set < (CALIBRATE_CYCLES + SAMPLES_TO_DISCARD))
    {   //Calibrate on startup, CALIBRATE_CYCLES cycles, do not place anything (tags, metal..) near the antenna during this period
        LED_14443A_ON;                     // if the led was on turn it off.  This is unnecessary as the port is reset in SetupWakeState
        __delay_cycles (100000);			//could be removed, just for indication that calibration is happening
        cal_set++;
        if ((time > SniffCardTimeThreshold) && (cal_set > SAMPLES_TO_DISCARD))
        {
            SniffCardTimeThreshold = time;   	// stores the maximum sense value received during calibration
        }
        if (cal_set >= (CALIBRATE_CYCLES + SAMPLES_TO_DISCARD))
        {
            SniffCardTimeThreshold += THRESHOLD_OFFSET;         // and add an offset, this new value will be used to determine if tag initially
        }
        ret_val = 1;	// calibration was performed
    }
    else
    {
    	ret_val = 0;	// calibration was not performed
    }

    return ret_val;
}
//Configure the ports for the most power efficient state
void SetupSleepState ()
{
    P4SEL = 0;
    P4DIR = 0;
    P4OUT = 0;
    P4REN = 0xff;
#ifdef UART_PRINTF
    P3SEL = BIT4 + BIT5;
    P3DIR = BIT4;
#else
    P3SEL = 0;
    P3DIR = 0x00;
#endif
    P3OUT = 0;
    P3REN = 0xFF;
    P2SEL = 0;
    P2DIR = 0x00;
    P2OUT = 0x08;
    P2REN = 0xFF;
    P1SEL = 0;
    P1OUT = 0;
    P1DIR = 0x00;
    P1REN = 0xFF;
    CAPD = 0;
    //SENSE_PIN_PULL_DOWN;
}
//Configure the ports for on state operation
void SetupWakeState ()
{
    //reset the original port settings
    P4SEL = 0;
    P4DIR = 0;
    P4OUT = 0;
    P4REN = 0xff;	//port four not used, keep it all pulled down in input state
#ifdef UART_PRINTF
    P3SEL = BIT4 + BIT5;
    P3DIR = BIT4;
    P3REN = 0xC0;
#else
    P3SEL = 0;
    P3DIR = 0x00;
    P3REN = 0xF0;
#endif
    P3OUT = 0x01;	//SPI peripheral on this port
    P2SEL = 0x00;
    P2DIR = 0x00;  //high-impedance ASK-OOK pin -bit2, also bit 0 MOD pin
    P2OUT = 0x00;
    P2REN = 0x00;
    P1SEL = 0x00;
    P1OUT = 0x00;
    P1DIR = 0xFD;
    P1REN = 0x0;

    IRQ_EDGE_SET;                       // rising edge interrupt

    LED_ALL_OFF;

    ENABLE_INTERRUPTS;                  // General enable interrupts

    MOD_TRF_DIR_Out;                    // Set up mod pin for output
    TRF_CARRIER_OFF;
#ifdef DEBUG
#define DEBUG_PIN_INIT	   	P3DIR |= BIT7
#define DEBUG_PIN_CLEAR		P3OUT &= ~BIT7
#endif

	//P2SEL |= BIT2;  // for debug purposes only - should be removed (routes comparator output to ASK pin)
	//P2DIR |= BIT2;  // for debug purposes only - should be removed (routes comparator output to ASK pin)

}
//Configures the timer B to wakeup the device from sleep
void SetWakeupTimer(u16_t num_cycles, u16_t clock_source)
{
    TBCTL = 0;
	TBCTL |= TBCLR;
	TBCTL &= ~TBCLR;				// reset the timer B
	TBCTL |= clock_source;				// ACLK (12k cycles per second)


	TBR = 0x0000;
	TBCCTL0 |= CCIE;				// compare interrupt enable
	TBCCR0 = num_cycles;
	TBCTL |=  MC0;       			// start counter in up mode (AK corrected)
}

//Stop the timer B
void StopWakeupTimer ()
{
	TBCTL &= ~(MC0 + MC1);
	TBR = 0;
	TBCCTL0 &= ~CCIE;				// compare interrupt enable
}

/* =======================================================================================================================
    Direct mode (no stop condition) ;
    Puts the TRF7970 into direct mode 0 with transmitter on at half power
 ======================================================================================================================= */
/*
void DirectMode(void)
{
    u08_t temp;
    if (SPIMODE) //SPI Mode
    {
        #ifndef SPI_BITBANG
        SLAVE_SELECT_LOW;  //Start SPI Mode
        while (!(IFG2 & UCB0TXIFG));            // USCI3_B0 TX buffer ready?
        UCB0TXBUF = 0x20 | CHIP_STATE_CONTROL;                  // Previous data to TX, RX

        while(UCB0STAT & UCBUSY)
        temp=UCB0RXBUF;


        while (!(IFG2 & UCB0TXIFG));            // USCI_B0 TX buffer ready?
        UCB0TXBUF = 0x40;                     // full-power, transmitter on, direct mode 0

        while(UCB0STAT & UCBUSY)
        temp=UCB0RXBUF;

        while (!(IFG2 & UCB0TXIFG));            // USCI_B0 TX buffer ready?
        UCB0TXBUF = 0x00;
        while(UCB0STAT & UCBUSY)
        temp=UCB0RXBUF;

        #endif
    }  //end of SPI Mode

} */
/* DirectMode */

//Sets up the TRF7970A for card sniffing
void InitForCardSniff()
{
    u08_t packet[2];

	packet[0] = MODULATOR_CONTROL;
	packet[1] = 0x30;                       // SYS_CLK is 13.56MHz
	Trf797xWriteSingle(packet, 2);

	packet[0] = ISO_CONTROL;
	packet[1] = 0x02;
    Trf797xWriteSingle(packet, 2);			// this register has to be written for proper operation
}

//Sets up the TRF7970A for standard operation
void ReInitForStandardOp(void)
{
    CAPD = 0;							// should not be used sooner. Only enable input buffer at card read and at power down

	Trf797xInitialSettings();			// Set MCU Clock Frequency to 6.78 MHz and OOK Modulation
    __delay_cycles (20000);				// 3ms at 6.78Mhz
    McuOscSel(TRF_CLOCK_SOURCE);       	// set the oscillator
}

//Detects if a card is nearby and if so returns a positive value (zero if failed)
u16_t ComparatorSlopeTime(void)
{
	u08_t command[2];

    InitComparator();

    InitTimerA();

    CACTL1 |= CAON;                 	// turn on the comparator
    CACTL1 |= CAIE;

    TagDetected = 0;

    //put TRF79x0A into active state
    command[0] = CHIP_STATE_CONTROL;
	command[1] = 0;
	Trf797xWriteSingle(command, 2);

	__delay_cycles (1500); 				// startup 100us

	//turn on the transmitter for about 20uS
	command[0] = CHIP_STATE_CONTROL;
	command[1] = 0x20;
	Trf797xWriteSingle(command, 2);


	__delay_cycles (100); 				// on for about 20us
    TACTL |= MC1;                       // SMCLK source, 13.56Mhz, div 1, timer starts count up

    //put the TRF79x0A into standby mode to conserve power
    command[0] = CHIP_STATE_CONTROL;
	command[1] = 0x80;
	Trf797xWriteSingle(command, 2);


    SetWakeupTimer (9000, SMCLK_CLOCK); // 660us.  This is a timeout if the tag is not detected for failsafe operation
    __bis_SR_register(LPM0_bits + GIE); // Ideally the comparator will wakeup the MSP430, if for some reason it does not, then the Timer_B will, preventing device halt

    StopWakeupTimer();					// Stop the timer if it has not timed out yet
    CACTL1 &= ~(CAON + CAIE);			// disable the comparator
    if (TagDetected == 0)
    {
        return 0;						// tag was not detected, return 0
    }
    else
    {
        return TAR;               		// Return the time before interrupt occurred
    }
}

//Initializes Timer A
void InitTimerA()
{
    TACTL = TASSEL_2;
    TAR = 0;
    TACTL |= TACLR;
    TACTL &= ~TACLR;
}

//Initializes the comparator to half internal reference
void InitComparator()
{
    CAPD = CAPD1;                   // Disables port driver, enables CA1

    CACTL1 = CAREF_2;               // .5VCC, interrupt enabled
    CACTL2 = CAF + P2CA1;           // P2.4 = CA1
}

// The initial sniff settings
void SniffInitialSettings(void)
{
	u08_t packet[2];

    packet[0] = SOFT_INIT;				// normal init sequence
	Trf797xDirectCommand (packet);
	packet[0] = IDLE;
	Trf797xDirectCommand (packet);

	packet[0] = REGULATOR_CONTROL;		// this is needed to be sent atmost 900uS after TRF79x0A to prevent the transmitter from turning on
	packet[1] = 0x00;
    Trf797xWriteSingle(packet, 2);

    packet[0] = CHIP_STATE_CONTROL;		// place the chip into standby modes
	packet[1] = 0x80;
	Trf797xWriteSingle(packet, 2);	    // TRF standby state;

	packet[0] = MODULATOR_CONTROL;
	packet[1] = 0x00;                   // SYS_CLK disabled to save power
	Trf797xWriteSingle(packet, 2);

	//packet[0] = FIFO_IRQ_LEVELS;		// not needed for this application - unless max FIFO writes are needed
	//packet[1] = 0x08;
    //Trf797xWriteSingle(packet, 2);
}


// Watchdog Timer interrupt service routine
#pragma vector=WDT_VECTOR
__interrupt void Watchdog_ISR(void)
{
    return;
}

//Comparator interrupt
#pragma vector=COMPARATORA_VECTOR
__interrupt void Comparator_A_ISR(void)
{
    //CACTL1 &= ~CAIFG;                     // clear any present interrupts from initialization
    TACTL &= ~(MC1+MC0);                    // stop the timer
    TagDetected = 1;

    __bic_SR_register_on_exit(LPM4_bits);   // exit LPM mode on exit
}
#pragma vector=TIMERB0_VECTOR
__interrupt void Timer_B0_ISR(void)
{
	TBCTL &= ~(MC0 + MC1);					// stop the timer
	TBR = 0;
	__bic_SR_register_on_exit(LPM4_bits);	// exit LMP on exit
}


